package com.mayi.fastdevelop.util;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.provider.MediaStore;
import android.text.TextUtils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * 文件管理工具
 */
public final class FileUtil {

    /**
     * 删除一个文件
     */
    public static void deleteFile(String filePath) {
        if (TextUtils.isEmpty(filePath))
            return;
        File file = new File(filePath);
        deleteFile(file);
    }

    /**
     * 删除文件
     */
    @SuppressWarnings("ResultOfMethodCallIgnored")
    public static void deleteFile(File file) {
        if (file == null) {
            return;
        }
        if (file.exists()) {
            if (file.isFile()) {
                file.delete();
            } else if (file.isDirectory()) {
                File files[] = file.listFiles();
                if (files != null) {
                    for (File subFile : files) {
                        deleteFile(subFile);
                    }
                }
            }
            file.delete();
        }
    }

    /**
     * 删除文件并刷新媒体库
     */
    public static void deleteFile(File file, Context context) {
        try {
            if (file.exists()) { // 判断文件是否存在
                if (file.isFile()) { // 判断是否是文件
                    file.delete(); // delete()方法 你应该知道 是删除的意思;
                    Uri localUri = Uri.fromFile(file);
                    String where = MediaStore.MediaColumns.DATA + "='" + file + "'";
                    context.getContentResolver().delete(MediaStore.Images.Media.EXTERNAL_CONTENT_URI, where, null);
                    Intent localIntent = new Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE, localUri);
                    context.sendBroadcast(localIntent);
                } else if (file.isDirectory()) { // 否则如果它是一个目录
                    File files[] = file.listFiles(); // 声明目录下所有的文件 files[];
                    if (files != null) {
                        for (int i = 0; i < files.length; i++) { // 遍历目录下所有的文件
                            deleteFile(files[i], context); // 把每个文件 用这个方法进行迭代
                        }
                    }
                }
                file.delete();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 获取SdCard路径
     */
    public static String getExternalStoragePath() {
        String state = Environment.getExternalStorageState();
        if (Environment.MEDIA_MOUNTED.equals(state) && Environment.getExternalStorageDirectory().canWrite()) {
            return Environment.getExternalStorageDirectory().getPath();
        }
        return null;
    }

    /**
     * 复制文件
     *
     * @param oldPath 源文件路径
     * @param newPath 目标文件路径
     */
    public static boolean copyFile(String oldPath, String newPath) {
        boolean isSuccess;
        InputStream is = null; // 输入原文件
        OutputStream os = null; // 输出目标文件文件
        try {
            File oldFile = new File(oldPath);
            if (oldFile.exists()) { // 文件存在时
                is = new BufferedInputStream(new FileInputStream(oldPath));
                os = new BufferedOutputStream(new FileOutputStream(newPath));
                final byte[] buffer = new byte[4096];
                int readLen;
                while ((readLen = is.read(buffer)) != -1) {
                    os.write(buffer, 0, readLen);
                }
                isSuccess = true;
            } else {
                isSuccess = false;
            }
        } catch (Exception e) {
            e.printStackTrace();
            isSuccess = false;
        } finally {
            closeQuietly(is, os);
        }
        return isSuccess;
    }

    /**
     * 关闭IO
     */
    public static boolean closeQuietly(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
                return true;
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }
        return true;
    }

    /**
     * 关闭多个IO
     */
    public static void closeQuietly(Closeable... closeables) {
        for (Closeable closeable : closeables) {
            closeQuietly(closeable);
        }
    }

    /**
     * 根据系统当前ms值生成文件名
     *
     * @param prefix 前缀缀名（例 b_）
     * @param suffix 后缀名（例 .jpg）
     */
    public static String getRandomFileName(String prefix, String suffix) {
        return prefix + System.currentTimeMillis() + suffix;
    }

    /**
     * 判断文件夹是否存在，不存在则创建
     */
    public static void isExistsDirFile(String filePath) {
        File dirFile = new File(filePath);
        if (!dirFile.exists()) {
            // 文件夹不存在直接创建
            dirFile.mkdirs();
        }
    }

    /**
     * 判断文件是否存在
     */
    public static boolean isExistsFile(String filePath) {
        if (TextUtils.isEmpty(filePath))
            return false;
        File file = new File(filePath);
        return file.exists();
    }

    public static boolean hasExtentsion(String filename) {
        final String extName = getExtensionName(filename);
        return !extName.isEmpty();
    }

    /**
     * 获取文件扩展名
     */
    public static String getExtensionName(String filename) {
        if ((filename != null) && (filename.length() > 0)) {
            int dot = filename.lastIndexOf('.');
            if ((dot > -1) && (dot < (filename.length() - 1))) {
                return filename.substring(dot + 1);
            }
        }
        return "";
    }


    /**
     * 获取文件名
     */
    public static String getFileNameFromPath(String filepath) {
        if ((filepath != null) && (filepath.length() > 0)) {
            int sep = filepath.lastIndexOf('/');
            if ((sep > -1) && (sep < filepath.length() - 1)) {
                return filepath.substring(sep + 1);
            }
        }
        return filepath;
    }


    /**
     * 获取sd卡剩余空间
     **/
    public static long getSDCardFreeSpaceSize() {
        if (!Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())) {
            return 0;
        }
        File path = Environment.getExternalStorageDirectory();
        StatFs sf = new StatFs(path.getPath());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            return sf.getAvailableBytes();
        } else {
            long singleBlockSize = sf.getBlockSize();
            long freeBlockCount = sf.getAvailableBlocks();
            return freeBlockCount * singleBlockSize;
        }
    }

    public static boolean makeDir(String dirPath) {
        File file = new File(dirPath);
        if (!file.exists()) {
            return file.mkdirs();
        } else {
            return true;
        }
    }

    public enum FileType {
        WORD,
        EXCEL,
        JPG,
        MP4,
        PPT,
        PDF,
        OTHER,
        TYPE_FOLDER,
        GIF
    }

    public static FileType getFileType(String name) {
        String exName = getExtensionName(name);
        exName = exName.toLowerCase();
        if (exName.equals("jpg") || exName.equals("png") || exName.equals("jpeg") || exName.equals("bmp")) {
            return FileType.JPG;
        } else if (exName.equals("mp4") || exName.equals("avi") || exName.equals("rmvb") || exName.equals("mov")) {
            return FileType.MP4;
        } else if (exName.equals("pptx") || exName.equals("ppt")) {
            return FileType.PPT;
        } else if (exName.equals("doc") || exName.equals("docx")) {
            return FileType.WORD;
        } else if (exName.equals("pdf")) {
            return FileType.PDF;
        } else if (exName.equals("xls") || exName.equals("xlsx")) {
            return FileType.EXCEL;
        } else if (exName.equals("gif")) {
            return FileType.GIF;
        }
        return FileType.OTHER;
    }

    /**
     * 根据byte数组，生成文件
     */
    public static File getFile(byte[] bfile, String filePath) {
        BufferedOutputStream bos = null;
        FileOutputStream fos = null;
        File file = null;
        try {
            File dir = new File(filePath);
            if (!dir.exists() && dir.isDirectory()) {//判断文件目录是否存在
                dir.mkdirs();
            }
            file = new File(filePath);
            fos = new FileOutputStream(file);
            bos = new BufferedOutputStream(fos);
            bos.write(bfile);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (bos != null) {
                try {
                    bos.close();
                } catch (IOException e1) {
                    e1.printStackTrace();
                }
            }
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e1) {
                    e1.printStackTrace();
                }
            }
        }
        return file;
    }


    /**
     * 使用系统当前日期加以调整作为照片的名称
     *
     * @param Prefix 前缀缀名（例 b_）
     * @param Suffix 后缀名（例 .jpg）
     */
    public static String getPhotoFileName(String Prefix, String Suffix) {
        return Prefix + System.currentTimeMillis() + Suffix;
    }

    public static boolean createFile(File file) {
        if (file != null && !file.exists()) {
            String absolutePath = file.getAbsolutePath();
            if (absolutePath.endsWith(File.separator)) {
                //创建文件夹
                if (!file.exists()) {//文件夹不存在
                    return file.mkdirs();
                }
            } else {
                //创建文件
                if (!file.exists()) {//文件不存在
                    if (!file.getParentFile().exists()) {
                        return file.getParentFile().mkdirs();
                    }
                }
            }
        }
        return false;
    }

    /**
     * 判断是否手机插入Sd卡
     */
    public static boolean hasSdcard() {
        return android.os.Environment.getExternalStorageState().equals(
                android.os.Environment.MEDIA_MOUNTED);
    }

    public static File getDCIMFile(String filePath, String imageName) {
        if (Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
            // 文件可用
            File dirs = new File(Environment.getExternalStorageDirectory(), "DCIM" + filePath);
            if (!dirs.exists()) dirs.mkdirs();
            File file = new File(Environment.getExternalStorageDirectory(), "DCIM" + filePath + imageName);
            if (!file.exists()) {
                try { //在指定的文件夹中创建文件
                    file.createNewFile();
                } catch (Exception e) {
                }
            }
            return file;
        } else {
            return null;
        }
    }
}
