package com.mayi.fastdevelop.view.dialog;

import android.annotation.SuppressLint;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;

import androidx.annotation.FloatRange;
import androidx.annotation.StyleRes;
import androidx.appcompat.app.AlertDialog;

import com.mayi.fastdevelop.R;


/**
 * 自定义dialog {@link # https://github.com/luoshihai/XXDialog/}
 * <p>
 * 用法:<BR/>
 * {@code new BaseDialog(this,R.layout.dialog)实现 convert()
 * .setCancelable......}
 * <p>
 * API:<BR/>
 * {@link #show()} 显示dialog <BR/>
 * {@link #show(boolean)}  如果为true 就显示默认的一个缩放动画<BR/>
 * {@link #showWithAnim(int)} 显示一个Dialog自定义一个弹出方式  具体怎么写 可以模仿上面的<BR/>
 * {@link #dismiss()}  dismiss dialog<BR/>
 * {@link #backgroundLight(float light)} 弹出时背景亮度 值为0.0~1.0    1.0表示全黑  0.0表示全白<BR/>
 * {@link #fromBottomToMiddle()} 从底部一直弹到中间<BR/>
 * {@link #fromBottom()} 从底部弹出 显示在底部<BR/>
 * {@link #fromLeftToMiddle()} 从左边一直弹到中间退出也是到左边<BR/>
 * {@link #fromRightToMiddle()} 从右边一直弹到中间退出也是到右边<BR/>
 * {@link #fromTop()} 从顶部弹出 从顶部弹出  保持在顶部<BR/>
 * {@link #fromTopToMiddle()} 从顶部谈到中间  从顶部弹出  保持在中间<BR/>
 * {@link #fullScreen()} 全屏显示<BR/>
 * {@link #fullWidth()} 全屏宽度<BR/>
 * {@link #fullHeight()} 全屏高度<BR/>
 * {@link #setWidthAndHeight(int width, int height)} 自定义宽高<BR/>
 * {@link #setOnKeyListener(DialogInterface.OnKeyListener onKeyListener)} 当dialog弹出是 按键的点击事件会被dialog获取<BR/>
 * {@link #setDialogDismissListener(DialogInterface.OnDismissListener)} 设置dismiss监听<BR/>
 * {@link #setOnCancelListener(DialogInterface.OnCancelListener)}  设置取消监听<BR/>
 * {@link #setCancelAble(boolean)} 设置能否被取消<BR/>
 * {@link #setCanceledOnTouchOutside(boolean cancel)} 设置点击其他地方能否被取消<BR/>
 * Created by ludas on 2017/5/5.
 */

public abstract class BaseDialog {
    private static Dialog mDialog;
    private Window mDialogWindow;
    private DialogViewHolder dialogHolder;
    private View mRootView;

    public BaseDialog(Context context, int layoutId) {
        dialogHolder = DialogViewHolder.get(context, layoutId);
        mRootView = dialogHolder.getConvertView();
        mDialog = new Dialog(context, R.style.dialog);
        mDialog.setContentView(mRootView);
        mDialogWindow = mDialog.getWindow();
        convert(dialogHolder);
    }

    /**
     * 把弹出框view holder传出去
     */
    public abstract void convert(DialogViewHolder holder);

    public static AlertDialog.Builder creatNormalDialogBuilder(Context context, String title, String message) {
        return new AlertDialog.Builder(context)
                .setTitle(title)
                .setMessage(message);
    }

    public static Dialog getDialog() {
        return mDialog;
    }

    /**
     * 显示dialog
     */
    public BaseDialog show() {
        if (mDialog != null && !mDialog.isShowing()) {
            mDialog.show();
        }
        return this;
    }

    /**
     * @param light 弹出时背景亮度 值为0.0~1.0    1.0表示全黑  0.0表示全白
     * @return
     */
    public BaseDialog backgroundLight(@FloatRange(from = 0, to = 1) float light) {
        if (light < 0.0 || light > 1.0)
            return this;
        WindowManager.LayoutParams lp = mDialogWindow.getAttributes();
        lp.dimAmount = light;
        mDialogWindow.setAttributes(lp);
        return this;
    }

    /**
     * 从底部一直弹到中间
     */
    @SuppressLint("NewApi")
    public BaseDialog fromBottomToMiddle() {
        mDialogWindow.setWindowAnimations(R.style.window_bottom_in_bottom_out);
        return this;
    }

    /**
     * 从底部弹出
     */
    public BaseDialog fromBottom() {
        fromBottomToMiddle();
        mDialogWindow.setGravity(Gravity.CENTER | Gravity.BOTTOM);
        return this;
    }

    /**
     * 从左边一直弹到中间退出也是到左边
     */
    public BaseDialog fromLeftToMiddle() {
        mDialogWindow.setWindowAnimations(R.style.window_left_in_left_out);
        mDialogWindow.addFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND);
        mDialogWindow.setGravity(Gravity.CENTER | Gravity.START);
        return this;
    }

    /**
     * 设置窗体的显示类型
     */
    public BaseDialog setWindowType() {
        mDialogWindow.setType(WindowManager.LayoutParams.TYPE_SYSTEM_ALERT);
        return this;
    }

    /**
     * 从右边一直弹到中间退出也是到右边
     *
     * @return
     */
    public BaseDialog fromRightToMiddle() {
        mDialogWindow.setWindowAnimations(R.style.window_right_in_right_out);
        mDialogWindow.addFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND);
        mDialogWindow.setGravity(Gravity.END);
        return this;
    }

    /**
     * 从顶部弹出 从顶部弹出  保持在顶部
     *
     * @return
     */
    public BaseDialog fromTop() {
        fromTopToMiddle();
        mDialogWindow.setGravity(Gravity.CENTER | Gravity.TOP);
        return this;
    }

    /**
     * 从顶部谈到中间  从顶部弹出  保持在中间
     *
     * @return
     */
    public BaseDialog fromTopToMiddle() {
        mDialogWindow.setWindowAnimations(R.style.window_top_in_top_out);
        mDialogWindow.addFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND);
        return this;
    }

    /**
     * @param style 显示一个Dialog自定义一个弹出方式  具体怎么写 可以模仿上面的
     * @return
     */
    public BaseDialog showWithAnim(@StyleRes int style) {
        mDialogWindow.setWindowAnimations(style);
        mDialog.show();
        return this;
    }

    /**
     * @param isAnimation 如果为true 就显示默认的一个缩放动画
     * @return
     */
    public BaseDialog show(boolean isAnimation) {
        mDialogWindow.setWindowAnimations(R.style.dialog_scale_animstyle);
        mDialog.show();
        return this;
    }

    /**
     * 全屏显示
     */
    public BaseDialog fullScreen() {
        WindowManager.LayoutParams wl = mDialogWindow.getAttributes();
        wl.height = ViewGroup.LayoutParams.MATCH_PARENT;
        wl.width = ViewGroup.LayoutParams.MATCH_PARENT;
        mDialog.onWindowAttributesChanged(wl);
        return this;
    }


    public BaseDialog setOnKeyListener(DialogInterface.OnKeyListener onKeyListener) {
        mDialog.setOnKeyListener(onKeyListener);
        return this;
    }

    /**
     * 全屏宽度
     */
    public BaseDialog fullWidth() {
        WindowManager.LayoutParams wl = mDialogWindow.getAttributes();
        wl.width = ViewGroup.LayoutParams.MATCH_PARENT;
        mDialog.onWindowAttributesChanged(wl);
        return this;
    }

    /**
     * 全屏高度
     */
    public BaseDialog fullHeight() {
        WindowManager.LayoutParams wl = mDialogWindow.getAttributes();
        wl.height = ViewGroup.LayoutParams.MATCH_PARENT;
        mDialog.onWindowAttributesChanged(wl);
        return this;
    }

    /**
     * @param width  自定义的宽度
     * @param height 自定义的高度
     * @return
     */
    public BaseDialog setWidthAndHeight(int width, int height) {
        WindowManager.LayoutParams wl = mDialogWindow.getAttributes();
        wl.width = width;
        wl.height = height;
        mDialog.onWindowAttributesChanged(wl);
        return this;
    }

    /**
     * cancel dialog
     */
    public void cancelDialog() {
        if (mDialog != null && mDialog.isShowing())
            dismiss();
    }

    /**
     * cancel dialog
     */
    public void dismiss() {
        if (mDialog != null && mDialog.isShowing()) {
            mDialog.dismiss();
        }
    }

    /**
     * 设置监听
     */
    public BaseDialog setDialogDismissListener(DialogInterface.OnDismissListener listener) {
        mDialog.setOnDismissListener(listener);
        return this;
    }

    /**
     * 设置监听
     */
    public BaseDialog setOnCancelListener(DialogInterface.OnCancelListener listener) {
        mDialog.setOnCancelListener(listener);
        return this;
    }

    /**
     * 设置是否能取消
     */
    public BaseDialog setCancelAble(boolean cancel) {
        mDialog.setCancelable(cancel);
        return this;
    }

    /**
     * 设置触摸其他地方是否能取消
     */
    public BaseDialog setCanceledOnTouchOutside(boolean cancel) {
        mDialog.setCanceledOnTouchOutside(cancel);
        return this;
    }
}
